<?php

namespace App\Entity;

use ApiPlatform\Metadata\Get;
use App\State\DepartmentStateProvider;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use ApiPlatform\Metadata\ApiResource;
use ApiPlatform\Metadata\GetCollection;
use App\Repository\DepartmentRepository;

#[ApiResource(
    operations: [
        new GetCollection(
            uriTemplate: 'department/list', // URI pour récupérer la liste des entities
            formats: ['json'],
            openapiContext: [
                'parameters' => [
                    ['in' => 'query', 'name' => 'nbPage', 'schema' => ['type' => 'integer'], 'required' => false], // Paramètre pour le numéro de page
                    ['in' => 'query', 'name' => 'itemsPerPage', 'schema' => ['type' => 'integer'], 'required' => false], // Paramètre pour les éléments par page
                    ['in' => 'query', 'name' => 'orderBy', 'schema' => ['type' => 'string', 'enum' => ["name"]], 'required' => false],
                    ['in' => 'query', 'name' => 'direction', 'schema' => ['type' => 'string', 'enum' => ['DESC', 'ASC']], 'required' => false],
                    ['in' => 'query', 'name' => 'name', 'schema' => ['type' => 'string'], 'required' => false],
                ],
            ],
            paginationEnabled: false, // Pagination désactivée
            deserialize: false, // Ne pas désérialiser le corps de la requête
            name: 'department-list',
            provider: DepartmentStateProvider::class, // Fournisseur pour gérer la logique
        ),
        new Get(
            uriTemplate: 'department/byCode',
            formats: ['json'],
            openapiContext: [
                'summary' => 'Get department by code',
                'parameters' => [
                    ['in' => 'query', 'name' => 'code', 'schema' => ['type' => 'string'], 'required' => true],
                ]
            ],
            name: 'department-by-code',
            provider: DepartmentStateProvider::class, // Fournisseur pour gérer la logique
        )
    ]
)]
#[ORM\Entity(repositoryClass: DepartmentRepository::class)]
class Department
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private ?int $id = null;

    #[ORM\Column(length: 255)]
    private ?string $code = null;

    #[ORM\Column(length: 255)]
    private ?string $name = null;

    /**
     * @var Collection<int, InseeRef>
     */
    #[ORM\OneToMany(targetEntity: InseeRef::class, mappedBy: 'departmentId')]
    private Collection $inseeRefs;

    public function __construct()
    {
        $this->inseeRefs = new ArrayCollection();
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getCode(): ?string
    {
        return $this->code;
    }

    public function setCode(string $code): static
    {
        $this->code = $code;

        return $this;
    }

    public function getName(): ?string
    {
        return $this->name;
    }

    public function setName(string $name): static
    {
        $this->name = $name;

        return $this;
    }

    /**
     * @return Collection<int, InseeRef>
     */
    public function getInseeRefs(): Collection
    {
        return $this->inseeRefs;
    }

    public function addInseeRef(InseeRef $inseeRef): static
    {
        if (!$this->inseeRefs->contains($inseeRef)) {
            $this->inseeRefs->add($inseeRef);
            $inseeRef->setDepartment($this);
        }

        return $this;
    }

    public function removeInseeRef(InseeRef $inseeRef): static
    {
        if ($this->inseeRefs->removeElement($inseeRef)) {
            // set the owning side to null (unless already changed)
            if ($inseeRef->getDepartment() === $this) {
                $inseeRef->setDepartment(null);
            }
        }

        return $this;
    }
}
