<?php

namespace App\Repository;

use App\Entity\Department;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;

/**
 * @extends ServiceEntityRepository<Department>
 */
class DepartmentRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Department::class);
    }

    public function getPaginatedDepartments(
        int $page,                     // Numéro de la page actuelle pour la pagination.
        int $itemsPerPage,             // Nombre d'éléments par page.
        ?string $orderBy,              // Champ par lequel trier les résultats (optionnel).
        ?string $direction,            // Direction du tri (ASC ou DESC).
        ?string $name              // Recherche par nom de department.
    ): array {
        // Création du QueryBuilder avec les IDs ajoutés
        $queryBuilder = $this->createQueryBuilder('d')
            ->select('d');

        // Filtre sur le nom de department
        if ($name !== null) {
            $sanitizedPatName = trim($name);
            $nameParts = explode(' ', $sanitizedPatName);
            foreach ($nameParts as $part) {
                $queryBuilder->andWhere(
                    $queryBuilder->expr()->orX(
                        $queryBuilder->expr()->like('d.name', ':part')
                    )
                )->setParameter('part', '%' . $part . '%');
            }
        }

        // Tri
        if ($orderBy == 'name' && $direction) {
            $queryBuilder->orderBy('d.name', $direction);
        }
        
        if ($itemsPerPage && $page) {
            // Compter le nombre total d'éléments
            $totalItems = count($queryBuilder->getQuery()->getArrayResult());
            // Calculer le nombre total de pages
            $totalPages = (int) ceil($totalItems / $itemsPerPage);
            // Pagination
            $queryBuilder
                ->setFirstResult(($page - 1) * $itemsPerPage)
                ->setMaxResults($itemsPerPage);
        }
        // Exécution de la requête et récupération des résultats sous forme d'un tableau associatif
        $results = $queryBuilder->getQuery()->getArrayResult();
        // Retourner les informations des patients sous forme de réponse JSON
        if ($itemsPerPage && $page) {
            $response = [
                'departments' => $results,
                'pagination' => [
                    'totalItems' => $totalItems,
                    'itemsPerPage' => $itemsPerPage,
                    'currentPage' => $page,
                    'totalPages' => $totalPages,
                ],
            ];
        } else {
            $response = [
                'departments' => $results,
            ];
        }

        return $response;
    }
}
