<?php

namespace App\State;

use App\Entity\Rpps;
use App\Entity\Department;
use Doctrine\DBAL\Exception;
use ApiPlatform\Metadata\Operation;
use ApiPlatform\State\ProviderInterface;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;


class DepartmentStateProvider implements ProviderInterface
{
    private $entityManager;
    private $requestStack;

    public function __construct(EntityManagerInterface $entityManager, RequestStack $requestStack)
    {
        $this->entityManager = $entityManager;
        $this->requestStack = $requestStack;
    }

    /*
     * Cette méthode récupère la liste des départements
     * à partir du repository Rpps en appelant la méthode listDepartments()
     */
    public function provide(Operation $operation, array $uriVariables = [], array $context = []): object|array|null
    {
        $request = $this->requestStack->getCurrentRequest();

        // Vérifie le nom de l'opération pour décider quelle méthode appeler
        if ($operation->getName() === 'listDepartments') {
            return $this->fetchDepartments();
        }

        if ($operation->getName() === 'department-list') {
            return $this->departmentList($request);
        }
        if ($operation->getName() === 'department-by-code') {
            return $this->departmentByCode($request);
        }
        return null;
    }
    public function fetchDepartments()
    {
        $departments = $this->entityManager->getRepository(Rpps::class)->listDepartments();

        //retourne un objet JsonResponse contenant la liste des départements sous forme de tableau,
        // qui sera envoyé en réponse à l'appel API.
        return new JsonResponse($departments);
    }
    public function departmentList($request)
    {
        try {
            // Récupération des paramètres de la requête
            $page = (int) $request->get('nbPage'); // Numéro de la page (par défaut : 1)
            $itemsPerPage = (int) $request->get('itemsPerPage'); // Nombre d'éléments par page (par défaut : 10)
            $orderBy = $request->get('orderBy'); // Critère de tri (par défaut : 'id')
            $direction = strtoupper($request->get('direction')); // Direction du tri (par défaut : 'ASC')
            $name = $request->get('name');

            // Vérification de la validité de la direction du tri
            if ($direction && !in_array($direction, ['ASC', 'DESC'], true)) {
                return new JsonResponse(
                    ['message' => 'Direction de tri invalide (valeurs autorisées : ASC, DESC)'],
                    Response::HTTP_BAD_REQUEST
                );
            }

            $departmentList = $this->entityManager->getRepository(Department::class)
                ->getPaginatedDepartments(
                    $page,           // Numéro de la page
                    $itemsPerPage,   // Nombre d'éléments par page
                    $orderBy,          // Critère de tri (optionnel)
                    $direction,          // Critère de tri (optionnel)
                    $name
                );

            // Retourner les informations des patients sous forme de réponse JSON
            return new JsonResponse(
                $departmentList,
                Response::HTTP_OK
            );
        } catch (Exception $e) {
            // Gestion des erreurs et retour d'un message d'erreur
            return new JsonResponse(
                ['message' => 'Une erreur est survenue : ' . $e->getMessage()],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }
    }

    public function departmentByCode($request): JsonResponse
    {
    $code = $request->get('code');
    $department = $this->entityManager->getRepository(Department::class)->findOneByCode($code);
    return new JsonResponse([
        'id' => $department?->getId(),
        'code' => $department?->getCode(),
        'name' => $department?->getName(),
    ]);
    }
}
