<?php

namespace App\State;

use ApiPlatform\Metadata\Operation;
use ApiPlatform\State\ProviderInterface;
use App\Entity\InseeRef;
use Symfony\Component\HttpFoundation\RequestStack;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\JsonResponse;

class InseeRefStateProvider implements ProviderInterface
{
    private $requestStack;
    private $entityManager;

    public function __construct(RequestStack $requestStack, EntityManagerInterface $entityManager)
    {
        $this->requestStack = $requestStack;
        $this->entityManager = $entityManager;
    }

    public function provide(Operation $operation, array $uriVariables = [], array $context = []): object|array|null
    {
        if ($operation->getName() === 'fetch-Insee-data') {
            return $this->fetchInseeData(); // Appel de la méthode pour récupérer les données Insee
        }

        return null;
    }

    public function fetchInseeData()
    {
        try {
            $request = $this->requestStack->getCurrentRequest();

            $birthDate = $request->query->get('BirthDate');
            $text      = $request->query->get('label');
            $page      = (int) $request->query->get('page', 1);
            $itemsPerPage = (int) $request->query->get('itemsPerPage', 10);

            $params = [
                'BirthDate' => $birthDate,
                'label'     => $text,
                'page'      => $page,
                'itemsPerPage' => $itemsPerPage,
            ];

            $result = $this->entityManager
                ->getRepository(InseeRef::class)
                ->fetchInseeData($params);

            // Calcul des métadonnées de pagination
            $total = $result['total'] ?? 0;
            $data = $result['data'] ?? [];

            return new JsonResponse([
                'message' => 'success',
                'data' => $data,
                'hydra:totalItems' => $total,
                'hydra:view' => [
                    'hydra:first' => '?page=1',
                    'hydra:last' => '?page=' . ceil($total / $itemsPerPage),
                    'hydra:previous' => $page > 1 ? '?page=' . ($page - 1) : null,
                    'hydra:next' => $page < ceil($total / $itemsPerPage) ? '?page=' . ($page + 1) : null,
                ]
            ]);
        } catch (\Exception $e) {
            return new JsonResponse([
                'message' => $e->getMessage(),
                'data' => []
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }
}
