<?php

namespace App\State;

use App\Entity\Rpps;
use Doctrine\DBAL\Exception;
use App\Service\ApiAddressService;
use ApiPlatform\Metadata\Operation;
use ApiPlatform\State\ProviderInterface;
use App\Entity\RppsPs;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RequestStack;

class RppsProvider implements ProviderInterface
{
    private $requestStack;
    private $entityManager;

    public function __construct(RequestStack $requestStack, EntityManagerInterface $entityManager)
    {
        $this->requestStack = $requestStack;
        $this->entityManager = $entityManager;
    }

    public function provide(Operation $operation, array $uriVariables = [], array $context = []): object|array|null
    {
        // Vérifie le nom de l'opération pour décider quelle méthode appeler
        if ($operation->getName() === 'fetch-ps-list') {
            return $this->fetchPsList(); // Appel de la méthode pour récupérer les communes
        }

        return null;
    }

    public function fetchPsList()
    {
        try {
            // Récupération des paramètres de la requête
            $request = $this->requestStack->getCurrentRequest();
            $psFullname = $request->query->get('psFullname');
            $ppNationalIdentification = $request->query->get('ppNationalIdentification');
            $speciality = $request->query->get('speciality');
            $department = $request->query->get('department');
            $phoneNumberCs = $request->query->get('phoneNumberCs');
            $addr = $request->query->get('addr');
            $firstnameExercise = $request->query->get('firstnameExercise');
            $exerciseName = $request->query->get('exerciseName');
            $itemsPerPage = min((int)$request->query->get('itemsPerPage', 10), 100);
            $page = max(1, (int)$request->query->get('page', 1));
            $orderBy = $request->query->get('orderBy', 'ps_fullname');
            $orderDirection = strtoupper($request->query->get('orderDirection', 'ASC'));

            // Champs autorisés alignés avec le repository
            $allowedOrderBy = ['ps_fullname', 'department', 'pp_national_identification', 'professional_category_code'];
            if (!in_array($orderBy, $allowedOrderBy)) {
                $orderBy = 'ps_fullname';
            }
            $specialities = $this->prepareFilterArray($speciality);
            // Clés des paramètres alignées avec le repository
            $params = [
                'psFullname' => $psFullname,
                'ppNationalIdentification' => $ppNationalIdentification,
                'specialities' => $specialities,
                'department' => $department,
                'phoneNumberCs' => $phoneNumberCs,
                'addr' => $addr,
                'firstnameExercise' => $firstnameExercise,
                'exerciseName' => $exerciseName,
                'page' => $page,
                'itemsPerPage' => $itemsPerPage,
                'orderBy' => $orderBy,
                'orderDirection' => $orderDirection
            ];

            // Appel au service pour récupérer les données
            $addresses = $this->entityManager->getRepository(RppsPs::class)->fetchPsList($params);

            // Add response headers for caching
            $response = new JsonResponse($addresses, Response::HTTP_OK);
            $response->headers->set('Cache-Control', 'public, max-age=300'); // 5 minutes cache
            $response->headers->set('Content-Type', 'application/json; charset=utf-8');
            return $response;
        } catch (\Exception $e) {
            return new JsonResponse([
                'message' => $e->getMessage(),
                'data' => []
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }
    private function prepareFilterArray($filter): array
    {
        // Si la donnée est une chaîne, la convertir en tableau
        if (!is_array($filter)) {
            $filter = explode(',', $filter);
            // Si la chaîne ne contient que des espaces ou est vide, retourner un tableau vide
            if (count($filter) === 1 && empty($filter[0])) {
                return [];
            }
        }
        return $filter;
    }
}
